#ifndef AHLGREN_BEST_FIRST_SEARCH
#define AHLGREN_BEST_FIRST_SEARCH

#include <queue> // std::priority_queue
#include <deque>
#include <algorithm>
#include <set>
#include <unordered_set>

#include "program.h"
#include "fitness.h"
#include "bottom.h"

namespace lp {

	// Define priority queue used for best-first search
	struct fitness_cmp { 
		bool operator()(const bsf_type& b, const bsf_type& c) const { return b.fitness() < c.fitness(); } 
	};
	typedef std::priority_queue<bsf_type,std::deque<bsf_type>,fitness_cmp> pqueue;

	// Hash a candidate made from bottom clause (with true_id) for redundancy checking

	struct candidate_hash {
		std::size_t operator()(const Functor& t) const {
			long long res = 0;
			std::map<id_type,int> vmap;
			// Skip hashing of true/0 literals
			long long base = 0;
			for (auto l = t.body_begin(); l != t.body_end(); ++l) {
				if (l->is_constant(true_id)) continue; // skip
				++base;
				res += base * hash_variant(t,vmap);
			}
			return std::size_t(res);
		}
	};

	struct candidate_variant
	{
		bool operator()(const Functor& s, const Functor& t) const {
			//std::cerr << "candidate_variant: " << *s << " <=> " << *t << "\n";
			auto i = s.body_begin();
			auto i_end = s.body_end();
			auto j = t.body_begin();
			auto j_end = t.body_end();
			std::map<id_type,id_type> vm1,vm2;
			auto skip = [](Functor::seq_const_iterator& it, Functor::seq_const_iterator end){
				while (it != end && it->is_constant(true_id)) ++it;
			};
			// Note: we must call is_variant on head to build variable bindings
			const bool bb = s.head()->is_renaming(t.head(),vm1,vm2);
			assert(bb);
			for (;; ++i,++j) {
				skip(i,i_end);
				skip(j,j_end);
				if (i == i_end) {
					if (j == j_end) {
						//std::cerr << "variants: " << *s << " === " << *t << "\n";
						return true;
					} else return false;
				}
				if (j == j_end) return false;
				if (!i->is_renaming(&*j,vm1,vm2)) return false;
			}
		}
	};

	stat Program::generalize_best_first(const sign_t& sign)
	{
		// Cache parameters
		const bool param_memo = params.is_set(parameters::memoize);
		const int param_csize = params.force_int(parameters::csize);
		const int param_noise = params.force_int(parameters::noise);
		const bool param_terminate = !params.is_unset(parameters::terminate);
		const int param_nodes = params.force_int(parameters::nodes);
		//const bool param_enforce_io_spec = params.is_set(parameters::enforce_io_spec);
		int nodes_left = param_nodes; // nodes left to explore
		int call_counter = 0;

		// Pick fitness function
		auto fitness = pick_fitness_function<bsf_type>(params);
		if (!fitness) {
			// Set default: compression
			fitness.reset( new fitness_compression<bsf_type>(params.force_int(parameters::terminate)) );
		}

		// These are the candidate that need to be evaluated first
		std::queue<bitstring> uneval;
		// Once uneval is empty, we expand the best using a priority queue
		pqueue qopen;
		// Once a node has been expanded, add it to closed
		std::unordered_set<Functor,candidate_hash,candidate_variant> closed;
		// Store mode I/O map
		//io_map iomap;

		auto init_search = [&]
		(Program& thy, Functor& bottom, int bsize, const std::vector<Mode>& modes, 
			std::list<clause>& pex, std::list<clause>& nex, Constraints&, stat& sts) -> bsf_type
		{
			fitness->init(bottom,modes,thy,pex,nex,sts[stat::exc]);
			// Increase bottom clause count
			++sts[stat::bottomc];
			// Compute and store mode structure
			//if (param_enforce_io_spec) iomap = make_io_map(bottom,modes);
			// Once all priority queue has no member clear()
			qopen = pqueue();
			// Start by evaluating top
			uneval = queue<bitstring>();
			bitstring bs(bsize,0); 
			uneval.push(bs);
			nodes_left = param_nodes - 1;
			mask_bottom(bottom,bs);
			// Clean up closed
			closed.clear();
			closed.insert(bottom);
			// Reset call counter
			call_counter = 0;
			return bsf_type();
		};


		auto find_candidate = [&]
		(Program& thy, std::list<clause>& pex, std::list<clause>& nex, Functor& bottom, int bsize, const std::vector<Mode>& modes, 
			const bsf_type& bsf, Constraints&, stat& sts, deadline_t deadline) -> bsf_type
		{
			//std::cerr << "Find_candidate, counter: " << ++call_counter << "\n";
			const int MAX_ITER = std::numeric_limits<int>::max(); //50000;
			int k = 0;
			//std::cerr << "\n";
			for ( ; uneval.empty() && k < MAX_ITER; ++k) {
				//std::cerr << "X";
				// Check deadline
				if (std::chrono::steady_clock::now() > deadline) {
					throw time_out();
				}
//expand:
				// Nothing more to evaluate, pick new best node to expand
				// Pick best in open (already evaluated) for expansion
				//std::cerr << "Qopen.size(): " << qopen.size() << "\n";
				if (qopen.empty()) throw search_aborted();
				bsf_type best = std::move(qopen.top());
				qopen.pop(); // erase it from priority queue
				// Can we expand this one?
				if (best.is_consistent(param_noise)) {
					continue; // Don't expand consistent nodes
				}
				const int ones = std::count(best.mask().begin(),best.mask().end(),1);

				assert(ones <= param_csize);
				if (ones < param_csize) {
					// Can make children
					// All children are given by adding a literal
					//std::cerr << "Making children, closed: " << closed.size() << "\n";
					bitstring pbs = best.mask();
					//const int MAX_CHILDREN = 20000; // only add first 10000 children TODO: parameter
					//int children_added = 0; 
					int k2 = 0;
					//std::cerr << "making children...";
					for (auto& b : pbs) {
						//std::cerr << "Y";
						// Check deadline
						if (std::chrono::steady_clock::now() > deadline) {
							throw time_out();
						}
						//if (++k2 > 50000) break; // TODO: parameter
						if (b == 0) {
							// Flip to 1 and store
							b = 1;
							// Check for revisit?
							bool ignore = false;
							if (param_memo) {
								mask_bottom(bottom,pbs);
								if (closed.find(bottom) != closed.end()) {
									// Revisit
									DEBUG_INFO(cout << "Skipping revisit: "; print_candidate(cout,bottom); cout << " (" << pbs << ")\n");
									ignore = true;
								}
							}
							if (!ignore) {
								// Add Child to Expansion Queue
								if (param_memo) {
									assert(closed.find(bottom) == closed.end());
									//std::cerr << "*** Inserting: " << pbs << "\n";
									auto p = closed.insert(bottom);
									assert(p.second);
									//std::cerr << "*********** Inserted :  " << **p.first << "\n";
									//cerr << "Search: " << *bottom << "\n";
									assert( closed.find(bottom) != closed.end() || !(cerr << "Missed: " << bottom << "\n"));
								}
								uneval.push(pbs);
								if (--nodes_left <= 0) break;
								//if (++children_added > MAX_CHILDREN) break; // TODO: parameter
							}
							// Flip back to 0
							b = 0;
						}
					}
					//std::cerr << "making children done\n";
				} // if ones < max_ones
			}

			if (k >= MAX_ITER) {
				// Failed to find candidate above, give up
				std::cerr << "WARNING: best-first search failed to produce candidate after " << MAX_ITER << " attempts\n";
				throw search_aborted();
			}

			bsf_type cand;
			bitstring bs = std::move(uneval.front());
			uneval.pop();
			assert(std::count(bs.begin(),bs.end(),1) <= param_csize);
			mask_bottom(bottom,bs);
			cand = std::move(bs);
			assert(closed.find(bottom) != closed.end() || !(std::cerr << "Candidate not yet added to closed: " << bottom << "\n"));
			//std::cerr << "Candidate: " << *bottom << " (" << is_mode_conformant(bottom,thy.params,modes) << ")\n";
			// Evaluate
			// Note: even if candidate is non-mode conformant, it needs to be evaluated since its expanded children may
			// be mode conformant. Expansion requires fitness (best-first), so evaluation is necessary.
			//std::cerr << "Evaluating fitness (" << thy.params[parameters::qres] << ")...";
			fitness->evaluate(bottom,cand,thy,pex,nex,bsf,sts[stat::exc],deadline);
			//std::cerr << "evaluation done\n";
			++sts[stat::fitc];
			if (cand.is_consistent(param_noise)) {
				++sts[stat::conc];
				// Check termination criterion
				if (param_terminate && cand.fitness() > 0) {
					throw cand;
				}
			} else {
				++sts[stat::inconc];
			}
			qopen.push(cand);

			return cand;
		};

		return this->generalize(sign,init_search,find_candidate);
	}


}


#endif

